'==============================================================================
' Copyright  Micromega Corporation 2006-2008. All rights reserved
'
' @file   	GCdistance.bs2
' @target	ARMmite/ARMexpress
'
' see: Application Note 39 - Calculating Great Circle Distances
'
' This program demonstrates the calculation of great circle distances.
' It reads the current latitude and longitude position from a GPS receiver,
' then calculates the great circle distance to a number of stored locations.
' The program can also be run without a GPS connection by selecting one of
' the pre-defined locations as the 'from' location.
'
' @author Cam Thompson, Micromega Corporation
' @version
'   March 31, 2008
'	- updated for ARMbasic V7
'	August 3, 2007
'	- ported to ARMmite/ARMexpress
'   July 25, 2007
'   - original version
'
'==============================================================================

'-------------------- uM-FPU V3.1 definitions ---------------------------------

#include "FPUspi.bas"					' select either SPI or I2C
'#include "FPUi2c.bas"

'==============================================================================
'==================== main definitions ========================================
'==============================================================================

'-------------------- uM-FPU Register Definitions -----------------------------
const	lat1		= 10				' uM-FPU register 10
const	long1		= 11				' uM-FPU register 11
const	lat2		= 12				' uM-FPU register 12
const	long2		= 13				' uM-FPU register 13

const	dist_nm		= 20				' distance - nautical miles
const	dist_km		= 21				' distance - kilometers
const	dist_miles	= 22				' distance - miles

'-------------------- uM-FPU Function Definitions -----------------------------
const	getID		= 0					' uM-FPU user function 0
const	getDistance	= 1					' uM-FPU user function 1
const	getLocation	= 2					' uM-FPU user function 2
const	radiansToDM	= 4					' uM-FPU user function 4
const	readNMEA	= 5					' uM-FPU user function 5
const	parseGPRMC	= 6					' uM-FPU user function 6
const	NMEA_Degrees= 7					' uM-FPU user function 7

'==============================================================================
'-------------------- main routine --------------------------------------------
'==============================================================================

Main:
	print
	print "Great Circle Distance"
		
	if Fpu_Reset <> SYNC_CHAR then		' reset FPU and check synchronization
		print "uM-FPU not detected"
		end
	else
		Print_Version					' display FPU version number
		print
	endif

	' check that user-defined functions are loaded
	'---------------------------------------------
	Fpu_Write5(CLR0, FCALL, getID, SELECTA, 0)
	Fpu_Wait
	Fpu_Write(LREADBYTE)
	if Fpu_Read <> 39 then
		print "GCdistance.fpu functions not loaded."
    	end
	endif

	' if from = 99, read current location from GPS
	' if from <> 99, use a stored location (0 to 7)
	' ---------------------------------------------
	from = 0

	print
	print "From:"
	print "  ";
	if from = 99 then
		print "Reading GPS ...";
		Fpu_Write3(FCALL, readNMEA, STRSET)
		Fpu_WriteString("Location")
	else
		Fpu_Write4(LONGBYTE, from, FCALL, getLocation)
		Fpu_Write3(COPY, lat2, lat1)
		Fpu_Write3(COPY, long2, long1)
	endif

  ' display latitude and longitude
  ' ------------------------------
	Fpu_Write(STRINS)
	Fpu_WriteString(" (")
	Fpu_Write5(SELECTA, lat1, FCALL, radiansToDM, STRINS)
	Fpu_WriteString(", ")
	Fpu_Write5(SELECTA, long1, FCALL, radiansToDM, STRINS)
	Fpu_WriteString(") ")
	Print_FpuString(READSTR)

  ' display distance to all other stored locations
  ' ----------------------------------------------
	print
	print "To:";

	for index = 0 TO 7
		if from <> index then
			print
			print "  ";
			Fpu_Write5(LONGBYTE, index, FCALL, getLocation, STRINS)
			Fpu_WriteString(" (")
			Fpu_Write5(SELECTA, lat2, FCALL, radiansToDM, STRINS)
			Fpu_WriteString(", ")
			Fpu_Write5(SELECTA, long2, FCALL, radiansToDM, STRINS)
			Fpu_WriteString(") ")
			Print_FpuString(READSTR)

			print
			print "   ";
			Fpu_Write5(SELECTA, dist_nm, FCALL, getDistance, FSET0)
			Print_Float(92)
			print " nautical miles"

			print "   ";
			Fpu_Write5(SELECTA, dist_km, FSET, dist_nm, FWRITE0)
			Fpu_WriteFloat($3FED0E56)
			Fpu_Write7(FMUL0, SELECTA, dist_miles, FSET, dist_km, FCNV, 13)
			Print_Float(92)
			print " miles";

			print
			print "   ";
			Fpu_Write2(SELECTA, dist_km)
			Print_Float(92)
			print" kilometers";
		endif
	next

Done:
	print
	print "Done."
	end
